<?php

/**
 * Reply odel.
 *
 * @since      3.24.0
 *
 * @package    Reviewer
 */

/**
 * Reply model.
 *
 * @package    Reviewer
 */
class RWP_Reply
{
    public static $submit_action_name = 'rwp_submit_reply';
    public static $delete_action_name = 'rwp_delete_reply';
    public static $edit_action_name = 'rwp_edit_reply';

    public static $post_meta_key = 'rwp_reply_';

    public static function get_submit_nonce()
    {
        return wp_create_nonce(static::$submit_action_name);
    }

    public static function get_edit_nonce()
    {
        return wp_create_nonce(static::$edit_action_name);
    }

    public static function get_delete_nonce()
    {
        return wp_create_nonce(static::$delete_action_name);
    }

    public static function on_submit_reply()
    {
        $nonce = isset($_POST['security']) ? $_POST['security'] : '';

        if (wp_verify_nonce($nonce, static::$submit_action_name) === false) {
            static::exit_with_bad_request();
        }

        // Get request body.
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $review_id = isset($_POST['review_id']) ? trim($_POST['review_id']) : '';
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : -1;
        $user_name = isset($_POST['user_name']) ? trim($_POST['user_name']) : '';
        $reply_comment = isset($_POST['reply_comment']) ? trim($_POST['reply_comment']) : '';

        // Post ID, Review ID.
        if (false === get_post_status($post_id) || empty($review_id) || $user_id == -1) {
            static::exit_with_bad_request();
        }

        // User.
        $user = wp_get_current_user();
        $user_id = (($user instanceof WP_User) && $user->ID == $user_id) ?  $user->ID : 0;

        if ($user_id <= 0 && empty($user_name)) {
            static::exit_with_error(__('Please type your name', 'reviewer'));
        }

        // Comment.
        if (empty($reply_comment)) {
            static::exit_with_error(__('Please write a comment', 'reviewer'));
        }

        $review_id = sanitize_text_field(stripslashes_deep($review_id));
        $user_name = $user_id <= 0 ? sanitize_text_field(stripslashes_deep($user_name)) : '';
        $reply_comment = implode("\n", array_map('sanitize_text_field', explode("\n", stripslashes_deep($reply_comment))));

        $reply = array(
            'reply_id' => uniqid('rwp_reply_'),
            'reply_review_id' => $review_id,
            'reply_post_id' => $post_id,
            'reply_user_id' => $user_id,
            'reply_author_name' => $user_name,
            'reply_comment' => $reply_comment,
            'reply_created_at' => current_time('timestamp'),
        );

        // Prepare the meta key.
        $key = static::$post_meta_key . $reply['reply_review_id'];

        // Save the reply.
        $res = add_post_meta($post_id, $key, $reply);

        // Check the saving process.
        if ($res === false) {
            wp_send_json_error(
                array(
                    'reply' => array(),
                    'feedback' => __('Error, unable to save the reply', 'reviewer')
                )
            );
        }

        // Prepare the response.
        $reply = static::prepare_reply($reply);

        wp_send_json_success(
            array(
                'reply' => $reply,
                'feedback' => __('Thank you!', 'reviewer')
            )
        );
    }

    public static function on_edit_reply()
    {
        global $wpdb;

        $nonce = isset($_POST['security']) ? $_POST['security'] : '';

        if (wp_verify_nonce($nonce, static::$edit_action_name) === false) {
            static::exit_with_bad_request();
        }

        // Authorization.
        if (!current_user_can('rwp_manage_user_reviews')) {
            static::exit_with_error(__('Anauthorized', 'reviewer'));
        }

        // Get request body.
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $review_id = isset($_POST['review_id']) ? trim($_POST['review_id']) : '';
        $reply_id = isset($_POST['reply_id']) ? trim($_POST['reply_id']) : '';
        $user_id = isset($_POST['reply_user_id']) ? intval($_POST['reply_user_id']) : -1;
        $user_name = isset($_POST['reply_author_name']) ? trim($_POST['reply_author_name']) : '';
        $reply_comment = isset($_POST['reply_comment']) ? trim($_POST['reply_comment']) : '';
        $reply_date = isset($_POST['reply_date']) ? trim($_POST['reply_date']) : '';

        // Post ID, Review ID.
        if (false === get_post_status($post_id) || empty($review_id) || $user_id == -1) {
            static::exit_with_bad_request();
        }

        // Find reply.
        $meta_key = 'rwp_reply_' . $review_id;

        $query = $wpdb->prepare('SELECT * FROM ' . $wpdb->postmeta . ' WHERE meta_key = %s && post_id = %d', $meta_key, $post_id);
        $post_metas = $wpdb->get_results($query, ARRAY_A);

        if (!is_array($post_metas)) {
            $post_metas = array();
        }

        $reply = null;

        foreach ($post_metas as $meta) {
            $r = maybe_unserialize($meta['meta_value']);

            if (
                isset($r['reply_id']) &&
                isset($r['reply_review_id']) &&
                isset($r['reply_post_id']) &&
                $r['reply_id'] == $reply_id &&
                $r['reply_review_id'] == $review_id &&
                $r['reply_post_id'] == $post_id
            ) {
                $reply = $r;
                $reply['rating_meta_id'] = $meta['meta_id'];
                break;
            }
        }

        // Reply not found.
        if ($reply == null) {
            static::exit_with_error(__('Reply not found', 'reviewer'));
        }

        // User.
        $user = get_user_by('ID', $user_id);
        $user_id = (($user instanceof WP_User) && $user->ID == $user_id) ?  $user->ID : 0;

        if ($user_id > 0) {
            $user_name = '';
        } else {
            $user_name = empty($user_name) ? $reply['reply_author_name'] : $user_name;
        }

        // Comment.
        if (empty($reply_comment)) {
            $reply_comment = $reply['reply_comment'];
        }

        $review_id = sanitize_text_field(stripslashes_deep($review_id));
        $user_name = sanitize_text_field(stripslashes_deep($user_name));
        $reply_comment = implode("\n", array_map('sanitize_text_field', explode("\n", stripslashes_deep($reply_comment))));

        // Date.
        $date = strtotime($reply_date, current_time('timestamp'));

        if ($date === false) {
            $date = intval($reply['reply_created_at']);
        }

        $reply_edited = array(
            'reply_id' => $reply['reply_id'],
            'reply_review_id' => $reply['reply_review_id'],
            'reply_post_id' => $reply['reply_post_id'],
            'reply_user_id' => $user_id,
            'reply_author_name' => $user_name,
            'reply_comment' => $reply_comment,
            'reply_created_at' => $date,
        );

        $res = $wpdb->update(
            $wpdb->postmeta,
            array('meta_value' => maybe_serialize($reply_edited)),
            array('meta_id'    => intval($reply['rating_meta_id'])),
            array('%s'),
            array('%d')
        );

        if ($res === false) {
            static::exit_with_error(__('Unable to update the reply', 'reviewer'));
        }

        // Prepare the response.
        $reply = static::prepare_reply($reply_edited);

        wp_send_json_success(
            array(
                'reply' => $reply,
                'feedback' => __('Done.', 'reviewer')
            )
        );
    }

    public static function on_delete_reply()
    {
        global $wpdb;

        $nonce = isset($_POST['security']) ? $_POST['security'] : '';
        $reply_id = isset($_POST['reply_id']) ? trim($_POST['reply_id']) : '';
        $review_id = isset($_POST['review_id']) ? trim($_POST['review_id']) : '';
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;

        // Validate the request.
        if (wp_verify_nonce($nonce, static::$delete_action_name) === false || empty($reply_id) || empty($review_id) || $post_id <= 0) {
            static::exit_with_bad_request();
        }

        // Authorization.
        if (!current_user_can('rwp_manage_user_reviews')) {
            static::exit_with_error(__('Anauthorized', 'reviewer'));
        }

        $meta_key = 'rwp_reply_' . $review_id;

        $query = $wpdb->prepare('SELECT * FROM ' . $wpdb->postmeta . ' WHERE meta_key = %s && post_id = %d', $meta_key, $post_id);
        $post_metas = $wpdb->get_results($query, ARRAY_A);

        if (!is_array($post_metas)) {
            $post_metas = array();
        }

        $reply = null;

        foreach ($post_metas as $meta) {
            $r = maybe_unserialize($meta['meta_value']);

            if (
                isset($r['reply_id']) &&
                isset($r['reply_review_id']) &&
                isset($r['reply_post_id']) &&
                $r['reply_id'] == $reply_id &&
                $r['reply_review_id'] == $review_id &&
                $r['reply_post_id'] == $post_id
            ) {
                $reply = $r;
                $reply['rating_meta_id'] = $meta['meta_id'];
                break;
            }
        }

        // Reply not found.
        if ($reply == null) {
            static::exit_with_error(__('Reply not found', 'reviewer'));
        }

        $res = $wpdb->delete($wpdb->postmeta, array('meta_id' => $reply['rating_meta_id']), array('%d'));

        if ($res === false || $res == 0) {
            static::exit_with_error(__('Unable to delete the reply', 'reviewer'));
        }

        wp_send_json_success(
            array(
                'feedback' => __('Done!', 'reviewer')
            )
        );
    }

    public static function exit_with_bad_request()
    {
        wp_send_json_error(
            array(
                'reply' => array(),
                'feedback' => __('Bad request', 'reviewer')
            )
        );
    }

    public static function exit_with_error($error)
    {
        wp_send_json_error(
            array(
                'reply' => array(),
                'feedback' => $error,
            )
        );
    }

    public static function get_review_replies($post_id, $review_id)
    {
        $post_id = intval($post_id);

        if ($post_id <= 0 || !is_string($review_id) || empty($review_id)) {
            return array();
        }

        $key = static::$post_meta_key . $review_id;
        $result = get_post_meta($post_id, $key);

        // Sort replies.
        usort($result, array('RWP_Reply', 'sort_replies'));

        // Prepare replies.
        $replies = array();
        foreach ($result as $reply) {
            if (!isset($reply['reply_id'])) {
                continue;
            }

            $reply = static::prepare_reply($reply);
            $replies[] = $reply;
        }

        return $replies;
    }

    private static function prepare_reply($reply)
    {
        // Get the preferences.
        $preferences = RWP_Reviewer::get_option('rwp_preferences');

        // Avatar.
        $reply['reply_author_avatar'] = RWP_User_Review::get_avatar_url($reply['reply_user_id'], 48);

        // Dispaly name.
        if ($reply['reply_user_id'] > 0) {
            $wp_user = get_user_by('id', $reply['reply_user_id']);
            $reply['reply_author_name'] = ($wp_user instanceof WP_User) ? $wp_user->display_name : __('Anonymous', 'reviewer');
        } else {
            $reply['reply_author_name'] = !empty($reply['reply_author_name']) ? $reply['reply_author_name'] : __('Anonymous', 'reviewer');
        }

        // Date.
        $human_format = (isset($preferences['preferences_users_reviews_human_date_format']) && $preferences['preferences_users_reviews_human_date_format'] == 'yes');

        $created_at = $reply['reply_created_at'];

        $reply['reply_date'] = date_i18n('Y-m-d H:i', intval($created_at));

        if ($human_format) {
            $reply['reply_created_at'] = sprintf(__('%s ago', 'reviewer'), human_time_diff(intval($created_at), current_time('timestamp')));
        } else {
            $date_format = get_option('date_format');
            $time_format = get_option('time_format');
            $reply['reply_created_at'] = date_i18n($date_format . ', ' . $time_format, intval($created_at));
        }
        return $reply;
    }


    public static function sort_replies($a, $b)
    {
        if (!isset($a["reply_created_at"]) || !isset($b["reply_created_at"])) {
            return 0;
        }

        if ($a["reply_created_at"] == $b["reply_created_at"])
            return 0;

        return ($a["reply_created_at"] > $b["reply_created_at"]) ? 1 : -1;
    }
}
